#!/usr/local/bin/perl
# 
# $Header: has/src/crsagents/appvipctl/appvipcfg.pl /main/12 2012/02/14 01:37:33 rbvarshn Exp $
# $Header: has/src/crsagents/appvipctl/appvipcfg.pl /main/12 2012/02/14 01:37:33 rbvarshn Exp $
#
# appvipcfg.pl
# 
# Copyright (c) 2008, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      appvipcfg.pl - Application VIP configuration utility
#
#    DESCRIPTION
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    rbvarshn    01/31/12 - Bug fix : 13498773
#    rakebhat    10/04/10 - appvipcfg delete option should delete resource type
#    rakebhat    09/29/10 - APPVIPCFG usage should contain failback option
#    hkanchar    02/20/10 - Fix compile error
#    hkanchar    01/26/10 - Add dependencies on cluster_net<> type
#    hkanchar    03/05/09 - Use crsconfig_lib.pm
#    hkanchar    11/18/08 - Check for type configured before adding the
#                           resource
#    hkanchar    09/02/08 - Creation
# 
use Getopt::Long;
use strict;
use English;
use Sys::Hostname;
use crsutils;
use File::Spec::Functions;
use Cwd;

require crsutils;

use vars qw ($help_text $opt_config $opt_deconfig $opt_networkresource $opt_vipresname 
$opt_ipaddress $opt_username $opt_groupname $opt_failback);

#Initialize constants used in the script

my $CRS_HOME;
my $APPVIPTYPE = "appvip";
my $APPVIPPREFIX = "app";
my $BASERESOURCETYPE = "cluster_vip";
my $TEMPLATEFILENAME = "appvip.type";
my $CRSCTL;
my $ORA_PREFIX = "ora";
my $TYPE_SUFFIX = "type";
my $NETWORKRESOURCE_SUFFIX = "network";
my $NETWORKNAMEPREFIX = "net";
my $RESOURCES_CONFIGURED = "";
my $TYPE_CONFIGURED = "";

our $DEBUG = 1;
our $CRSCFG_TRACE = 1;
our $CRSCFG_TRACE_FILE;


#subroutine declarations



# Begin Main program


#Command line arguments
@ARGV or (&print_usage and exit(1));

#Crs Home needs to be set in the environment
check_crshome();

# Get Command line options
get_options();

#Banner
if (! $help_text)
{
  print_banner();
}

# Process command line arguments
process_arguments();

# End Main Program

####------------------------------------------------------------------------
#### Main Function for processing the command line arguments 
# ARGS 0 
sub process_arguments
{
  if (($help_text) || (lc($ARGV[0]) eq "-h"))
  {
     print_usage();
     exit 1;
  }
  elsif (($opt_config) || (lc($ARGV[0]) eq "create"))
  {
    if ($opt_ipaddress eq "" || $opt_vipresname eq "" ||
        $opt_networkresource eq "" || $opt_username eq "") 
    {
      print_usage();
      exit 1;
    }
    else
    {
      config();
    }
  } 
  elsif (($opt_deconfig) || (lc($ARGV[0]) eq "delete"))
  {
    if ($opt_vipresname eq "")
    {
      print_usage();
      exit 1;
    }
    else
    {
      deconfig();
    }
  }
  else
  {
    print_usage();
    exit 1;
  }
}

####------------------------------------------------------------------------
#### Function for printing the usage 
# ARGS 0 
sub print_usage
{
print
q{
  Usage: appvipcfg create -network=<network_number> -ip=<ip_address> -vipname=<vipname>
                          -user=<user_name>[-group=<group_name>] [-failback=0 | 1]
                   delete -vipname=<vipname>
}
}

####------------------------------------------------------------------------
#### Function for printing the banner 
# ARGS 0 

sub print_banner()
{
  print "Production Copyright 2007, 2008, Oracle.All rights reserved\n";
}

####------------------------------------------------------------------------
#### Function to get the value of ORA_CRS_HOME from the OLR File 
# ARGS 0 

sub check_crshome()
{
  $CRS_HOME = $ENV{"ORA_CRS_HOME"};
  $CRSCTL = catfile ($CRS_HOME, "bin", "crsctl");
  if ($CRS_HOME eq "") 
  {
    print("ORA_CRS_HOME is not set in the environment \n");
    exit 1;
  }   
  
}


####------------------------------------------------------------------------
#### Function for configuring the type and resource information
#### Uses the global $opt_vipresourcename , $opt_networkresource,
#### $opt_ipaddress, $opt_username, $opt_groupname
#### Tries to create the type along with the resourcename
#### Type Name :- app.appvip.type
#### Res  Name :- Passed as a command line argument 
# ARGS : 0

sub config
{
  my $networkname = $NETWORKNAMEPREFIX.$opt_networkresource;
  my $vipresourcename = $opt_vipresname;
  my $ipaddress = $opt_ipaddress;
  my $username = $opt_username;
  my $groupname = $opt_groupname;
  my $acllist   = "";
  my $hostingmember = "";

  

  my $appviptype = $APPVIPPREFIX.".".$APPVIPTYPE."_".$networkname.".".$TYPE_SUFFIX;
  my $netresname = $ORA_PREFIX.".".$networkname.".".$NETWORKRESOURCE_SUFFIX;
  my $typefile = $CRS_HOME."/"."crs"."/"."template"."/".$TEMPLATEFILENAME;
  my $brestype = $ORA_PREFIX.".".$BASERESOURCETYPE."_".$networkname.".".$TYPE_SUFFIX;

  if ($^O eq "MSWin32")
  {
    $acllist = "'owner:$username:rwx,pgrp::r-x,other::r--'";
  }
  else
  {
    if ($opt_groupname eq "") 
    {
      $acllist = "'owner:root:rwx,pgrp:root:r-x,other::r--,".
                  "user:$username:r-x'";
    }
    else
    {
      $acllist = "'owner:root:rwx,pgrp:root:r-x,other::r--,".
                  "group:$groupname:r-x,user:$username:r-x'";
    }
  }

  type_configured($appviptype);
  if (!($TYPE_CONFIGURED))
  {
    trace("Creating Resource Type");
    system_cmd($CRSCTL,"add", "type", $appviptype, "-basetype", 
               $brestype, "-file", $typefile);

  }
  else
  {
    trace("Skipping type creation");
  }
  $hostingmember = hostname();
  trace("Create the Resource");
  my @appsvipattrib = (
                 "\"USR_ORA_VIP=$ipaddress",
                 "START_DEPENDENCIES=hard\($netresname\) pullup\($netresname\)",
                 "STOP_DEPENDENCIES=hard\($netresname\)",
                 "ACL=$acllist",
                 "HOSTING_MEMBERS=$hostingmember",
                 "APPSVIP_FAILBACK=$opt_failback\""
                 );
  system_cmd($CRSCTL, "add", "resource", $vipresourcename, "-type", 
             $appviptype, "-attr",join(',', @appsvipattrib));

}

####---------------------------------------------------------
#### Function for cleaning up our configruation
#### Uses the global $opt_vipresourcename
#### Tries to remove the type along with the resourcename
#### If a resource is running, resource deletion will fail, If a
#### resource of this particular type is running, type deletion will
#### fail
# ARGS : 0

sub deconfig
{

  my $vipresname = $opt_vipresname;
  my $appviptype = "";

  get_resource_type($vipresname);
  if ($RESOURCES_CONFIGURED)
  {
    $appviptype = $RESOURCES_CONFIGURED;
  }

  trace("Deleting the resource");
  system_cmd($CRSCTL,"delete", "res", $vipresname);
  if($appviptype)
  {
    trace("Removing the type");
    system_cmd($CRSCTL, "delete", "type", $appviptype);
  }

}

####---------------------------------------------------------
#### Function checks if a resource  is configured for a given type
# ARGS : 1

sub get_resource_type
{
  my $vipresname = $_[0];
  $RESOURCES_CONFIGURED="";

  my $command = "$CRSCTL stat res '$vipresname' |";
  open( STATCMD, $command);
  while (<STATCMD>)
  {
    if(!index($_, "TYPE"))
    {
      my  @saveoutput = split(/=/);
      $RESOURCES_CONFIGURED = $saveoutput[1];
      last;
    }
  }
}

####---------------------------------------------------------
#### Function checks if a type  is configured
# ARGS : 1

sub type_configured
{
  my $appviptype = $_[0];
  my $output;
  $TYPE_CONFIGURED = "";
  my $command = "$CRSCTL stat type $appviptype |";
  open( STATCMD, $command);
  while (<STATCMD>)
  {
    $output = $_;
    if ( $output =~m/TYPE_NAME=app.appvip.type/)
    {
      $TYPE_CONFIGURED = $_;
      last;
     }
  }
}

####---------------------------------------------------------
#### Function for reading the command line arguments and
#### Setting the location for the log files
# ARGS : 0

sub get_options()
{
  GetOptions("create"=>\$opt_config,
             "delete"=>\$opt_deconfig,
             "-h" =>\$help_text,
             "vipname=s"=>\$opt_vipresname,
             "ip=s"=>\$opt_ipaddress,
             "network=i"=>\$opt_networkresource,
             "failback=i"=>\$opt_failback,
             "user=s"=>\$opt_username,
             "group=s"=>\$opt_groupname);

  my $host = hostname(); 
  $CRSCFG_TRACE_FILE = catfile($CRS_HOME, "log",$host,"agent","crsd",
                             "appvipcfg_$PID.log");
}
